#!/bin/sh

SSH_OPTIONS="-o LogLevel=quiet -o ServerAliveInterval=1 -o ConnectTimeout=1 -o StrictHostKeyChecking=no"
EPC_IP="192.168.10.10"
LOG_FILE=""

RED='\033[0;31m'
GREEN='\033[0;32m'
NC='\033[0m' # No Color


set_path_variables()
{
    if [[ -d "/usr/C15/" ]] ; then  # old structure

        PATH_install_update_sh=/usr/C15/scripts/install-update.sh
        userlandCheck="ls /usr/local/ | grep C15"
        PATH_text2soled=/usr/C15/text2soled/text2soled
        printf "Detected *old* project structure\n" | tee -a $LOG_FILE

    else  # new structure

        PATH_install_update_sh=/usr/bin/install-update.sh
        userlandCheck="ls /usr/local/bin | grep c15"
        PATH_text2soled=/usr/bin/text2soled
        printf "Detected *new* project structure\n" | tee -a $LOG_FILE

    fi
}

executeAsRootWithoutReporting() {
    if echo "sscl" | sshpass -p 'sscl' ssh ${SSH_OPTIONS} sscl@$EPC_IP "sudo -S /bin/bash -c '$1'" &> /dev/null; then
        return 0
    fi
    return 1
}

executeAsRoot() {
    if executeAsRootWithoutReporting "$2"; then
        if ! [[ -z $1 ]]; then
            echo -e "$1 ${GREEN}Yes!${NC}" | tee -a $LOG_FILE
        fi
        return 0
    fi

    if ! [[ -z $1 ]]; then
        echo -e "$1 ${RED}No!${NC}" | tee -a $LOG_FILE
    fi
    return 1
}

copyFilesToEpc() {
    sshpass -p 'sscl' scp $1 sscl@$EPC_IP:$2 &> /dev/null
    return $?
}

getInfoAsRoot() {
    echo "sscl" | sshpass -p 'sscl' ssh ${SSH_OPTIONS} sscl@$EPC_IP "sudo -S /bin/bash -c '$1'"
}

check_userland_exists() {
    executeAsRoot \
        "ePC Userland exists?" \
        "${userlandCheck}"
}

check_soundcard_driver_param() {
    executeAsRoot \
        "ePC Soundcard driver parameter fix applied?" \
        "cat /etc/modprobe.d/snd_hda_intel.conf | grep 'snd_hda_intel bdl_pos_adj=1,1,1'"
    printf "\n" | tee -a $LOG_FILE
}

check_epc_network_exists() {
    executeAsRoot \
        "ePC Accesspoint is configured?" \
        "test -e /etc/NetworkManager/system-connections/C15.nmconnection"
}

check_isolcpus() {
    if executeAsRootWithoutReporting "cat /proc/cpuinfo | grep i5-7260U"; then
        executeAsRoot \
            "ePC has isolated CPUS for realtime processes?" \
            "cat /proc/cmdline | grep isolcpus=domain,managed_irq,2-3"
    else
        executeAsRoot \
            "ePC has isolated CPUS for realtime processes?" \
            "cat /proc/cmdline | grep isolcpus"
    fi
}

check_mitigations() {
    executeAsRoot \
        "ePC has mitigations switched off?" \
        "cat /proc/cmdline | grep mitigations"
}

check_overlay() {
    executeAsRoot \
        "ePC OverlayFS order is correct?" \
        "mount | grep overlay | grep /nloverlay/os-overlay:/lroot"
}

check_usb_stick() {
    printf "\n" | tee -a $LOG_FILE

    executeAsRoot \
        "sshfs mount of BBBs usb-stick active on ePC?" \
        "mount | grep '/mnt/usb-stick type fuse. sshfs'"

    while ! mount | grep /mnt/usb-stick > /dev/null; do
        echo -e "${RED}Please connect a USB stick to the BBB!${NC}"
        sleep 2
    done

    sleep 5

    rm -f /mnt/usb-stick/test-file

    executeAsRoot \
        "Mount usb stick on ePC via sshfs?" \
        "sshfs -o allow_other,default_permissions root@192.168.10.11:/mnt/usb-stick/ /mnt/usb-stick"

    sleep 3

    executeAsRoot \
        "Write to usb stick on ePC?" \
        "echo '1' > /mnt/usb-stick/test-file"

    sleep 5

    if [ "$(cat /mnt/usb-stick/test-file)" = "1" ]; then
        echo -e "Could access file on usb stick that has been written by ePC via sshfs? ${GREEN}Yes!${NC}"
    else
        echo -e "Could access file on usb stick that has been written by ePC via sshfs? ${RED}No!${NC}"
    fi

   echo '2' > /mnt/usb-stick/test-file

   sleep 5

   executeAsRoot \
        "Read from usb stick on ePC?" \
        "[ \"\$(cat /mnt/usb-stick/test-file)\" = '2' ]"
    printf "\n" | tee -a $LOG_FILE

    executeAsRoot \
        "Unmount usb stick on ePC via sshfs?" \
        "umount /mnt/usb-stick"
}

check_BBB_AP() {
    if executeAsRoot "Is ePC1?" "cat /proc/cpuinfo | grep -e \"i3-5010\" -e \"i3-7100\""; then
        if iw wlan0 info | grep ssid | grep -v "_BBB" > /dev/null; then
            echo -e "BBB Accesspoint is active and is correctly missing the _BBB suffix? ${GREEN}Yes!${NC}"
            return 0
        fi

        echo -e "BBB Accesspoint is active and is correctly missing the _BBB suffix? ${RED}No!${NC}"
        return 1
    fi

    if iw wlan0 info | grep ssid | grep "_BBB" > /dev/null; then
        echo -e "BBB Accesspoint is active and has correct _BBB suffix? ${GREEN}Yes!${NC}"
        return 0
    else
        echo -e "BBB Accesspoint is active and has correct _BBB suffix? ${RED}No!${NC}"
        return 1
    fi

}

check_EPC_AP() {
    if ! executeAsRoot "Is ePC1?" "cat /proc/cpuinfo | grep -e \"i3-5010\" -e \"i3-7100\""; then # not ePC1
        if executeAsRoot "ePC Has Wifi device named wlan?" "ip a | grep wlan"; then
            if executeAsRoot "Is AP active on ePC?" "nmcli con show --active | grep C15"; then
                echo -e "ePC has AP enabled? ${GREEN}Yes!${NC}"
                return 0
            fi
        fi
        echo -e "ePC has AP enabled? ${RED}No!${NC}"
        return 1
    fi
}

check_iface_names() {
    executeAsRoot \
        "ePC Ethernet device is named eth0?" \
        "ip a | grep eth0"

    if ! executeAsRoot "Is ePC1" "cat /proc/cpuinfo | grep -e \"i3-5010\" -e \"i3-7100\""; then # not ePC1
      executeAsRoot \
          "ePC WiFi device is named wlan0?" \
          "ip a | grep wlan0"
    fi
}

check_ping_timeout() {
    executeAsRoot \
        "ePC Ping timeout workaround is installed in initramfs?" \
        "mkdir -p /mnt/sda1; mount /dev/sda1 /mnt/sda1; zstdcat /mnt/sda1/initramfs-linux-rt.img | grep ping_bbb"
}

get_date() {
    printf "\nePC - date: $(getInfoAsRoot 'date "+%F %T"')\n" | tee -a $LOG_FILE \
        && printf "BBB - date: $(date "+%F %T")\n" | tee -a $LOG_FILE
}

get_ro_mnts() {
    printf "\nePC - RO mounts:\n$(getInfoAsRoot 'mount | grep "(ro"')\n" | tee -a $LOG_FILE \
        && printf "\nBBB - RO mounts:\n$(mount | grep "(ro")\n" | tee -a $LOG_FILE
}

get_journal_dmesg() {
    rm /tmp/epc_journal.log /tmp/epc_dmesg.log /tmp/bbb_journal.log /tmp/bbb_dmesg.log > /dev/null

    getInfoAsRoot 'journalctl -b -p "emerg".."warning"' >> /tmp/epc_journal.log \
        && getInfoAsRoot 'dmesg --level=err,warn' >> /tmp/epc_dmesg.log \
        && printf "\nGot journal and dmesg from ePC: ${GREEN}Yes!${NC}\n" | tee -a $LOG_FILE \
        || printf "\nGot journal and dmesg from ePC: ${RED}No!${NC}\n" | tee -a $LOG_FILE

    journalctl -b -p "emerg".."warning" >> /tmp/bbb_journal.log \
        && dmesg --level=err,warn >> /tmp/bbb_dmesg.log \
        && printf "Got journal and dmesg from BBB: ${GREEN}Yes!${NC}\n" | tee -a $LOG_FILE \
        || printf "Got journal and dmesg from BBB: ${RED}No!${NC}\n" | tee -a $LOG_FILE

    return 0
}

get_partition_info() {
    printf "\nePC - lsblk:\n$(getInfoAsRoot 'lsblk')\n" | tee -a $LOG_FILE \
        && printf "\nePC - mount:\n$(getInfoAsRoot 'mount | grep /dev/sda')\n" | tee -a $LOG_FILE

    printf "\nBBB - lsblk:\n$(lsblk)\n" | tee -a $LOG_FILE \
        && printf "\nBBB - mount:\n$(mount | grep /dev/mmc)\n" | tee -a $LOG_FILE
    return 0
}

check_persistent_integrity() {
    printf "\n" | tee -a $LOG_FILE
    executeAsRoot \
        "" \
        "systemctl stop playground ; umount /persistent"
    executeAsRoot \
        "ePC file system on /persistent OK?" \
        "fsck -f -n /dev/sda4"
    getInfoAsRoot "fsck -f -n /dev/sda4"
    executeAsRoot \
        "" \
        "mount /dev/sda4 /persistent ; systemctl start playground"
    printf "\n" | tee -a $LOG_FILE
}

check_persistent_files() {
    printf "\n" | tee -a $LOG_FILE
    executeAsRoot \
        "ePC Calibration files?" \
        "ls /persistent/calibration/*.cal"
    executeAsRoot \
        "Calibration .bin file?" \
        "ls /persistent/calibration/calibration.bin"
    executeAsRoot \
        "Calibration .ini file?" \
        "ls /persistent/calibration/calibration.ini"
    getInfoAsRoot "cat /persistent/calibration/calibration.ini"
    executeAsRoot \
        "ePC Settings.xml file?" \
        "ls /persistent/settings.xml"

    printf "\n" | tee -a $LOG_FILE
}

check_alsa_settings() {
    executeAsRoot \
        "ePC Alsamixer set to 100%?" \
        "amixer -D hw:PCH sget 'Master' | grep 100%"
    printf "\n" | tee -a $LOG_FILE
}

get_network_settings_bbb() {
    cat /etc/network/interfaces | tee -a $LOG_FILE
    cat /etc/hostapd.conf | tee -a $LOG_FILE
}

get_update_script_vers() {
    VERS=$(cat ${PATH_install_update_sh} | grep "version :")
    printf "\nUpdate script ${VERS}\n\n" | tee -a $LOG_FILE
}

check_for_present_tools() {
    for i in sshpass rsync socat thttpd playcontroller mke2fs; do
        if command -V "$i" &> /dev/null; then
            printf "$i installed: ${GREEN}Yes!${NC}\n" | tee -a $LOG_FILE
        else
            printf "$i installed: ${RED}No!${NC}\n" | tee -a $LOG_FILE
        fi
    done

    if [[ -x ${PATH_text2soled} ]]; then
        printf "text2soled installed: ${GREEN}Yes!${NC}\n" | tee -a $LOG_FILE
    else
        printf "text2soled installed: ${RED}No!${NC}\n" | tee -a $LOG_FILE
    fi

    if [[ -x /nonlinear/scripts/mxli ]]; then
        printf "mxli installed: ${GREEN}Yes!${NC}\n" | tee -a $LOG_FILE
    else
        printf "mxli installed: ${RED}No!${NC}\n" | tee -a $LOG_FILE
    fi

    printf "\n" | tee -a $LOG_FILE
    return 0
}

check_lpc() {
    systemctl stop bbbb

    printf "\nLPC Hardware ID: $(lpc req uhid64; lpc-read -a +6 +q | sed -n -e 's/^.*: //p') \n" | tee -a $LOG_FILE \
        || printf "${RED}Can not get HW ID!${NC}\n" | tee -a $LOG_FILE

    printf "\nLPC Software Version: $(lpc req sw-version; lpc-read -a +n +q | sed -n -e 's/^.*Version: //p') \n" | tee -a $LOG_FILE \
        || printf "${RED}Can not get LPC version!${NC}\n" | tee -a $LOG_FILE

    printf "LPC Calibration Status: $(lpc req at-status; lpc-read -a +n +q | sed -n -e 's/^.*: //p') \n" | tee -a $LOG_FILE \
        || printf "${RED}Can not get Calibration Status!${NC}\n" | tee -a $LOG_FILE

    lpc reset && printf "LPC reset: ${GREEN}Yes!${NC}\n" | tee -a $LOG_FILE \
        || printf "LPC reset: ${RED}No!${NC}\n" | tee -a $LOG_FILE

    printf "\n${GREEN}"
    cat /usr/bin/playcontroller | grep C15
    printf "${NC}"

    systemctl restart bbbb
    lpc reset > /dev/null  # this reset with BBBB on causes proper parameter/calibration upload to LPC after reset
    return 0
}

get_uuids_and_info() {
    BBB_ID=$(hexdump -s 16 -n 12 -C -v /sys/devices/platform/ocp/44e0b000.i2c/i2c-0/0-0050/eeprom | awk -F "|" '{print $2}')
    printf "\nBBB ID: ${BBB_ID}\n" | tee -a $LOG_FILE

    printf "ePC info:\n$(getInfoAsRoot 'NLnucInfo.sh')\n" | tee -a $LOG_FILE

    return 0
}

run_epc_stress_test() {
    TEST_DURATION=$1
    CHECK_FREQ=$2

    if [[ "$3" == "S" ]] ; then  # stress test with pre-installed 'mprime'

        executeAsRoot "Turbo Boost off?" "echo 1 > /sys/devices/system/cpu/intel_pstate/no_turbo"
        getInfoAsRoot "cat /sys/devices/system/cpu/intel_pstate/no_turbo"
        printf "Will run stress test on ePC for ${TEST_DURATION}s\n" | tee -a $LOG_FILE
        getInfoAsRoot "/usr/local/C15/scripts/mprime -t &> /dev/null &"
        executeAsRoot "Benchmark started?" "pidof mprime"
        STRESS_TEST_STARTED=true

        for COUNTER in $(seq 0 $CHECK_FREQ $TEST_DURATION); do
            TEMP_CORE_0=$(getInfoAsRoot 'sensors | grep "Core 0:" | cut -d"+" -f2 | cut -d"." -f1')
            [ $TEMP_CORE_0 -ge 85 ] && printf "Core 0 critical: $TEMP_CORE_0\n" | tee -a $LOG_FILE && break

            TEMP_CORE_1=$(getInfoAsRoot 'sensors | grep "Core 1:" | cut -d"+" -f2 | cut -d"." -f1')
            [ $TEMP_CORE_1 -ge 85 ] && printf "Core 1 critical: $TEMP_CORE_1\n" | tee -a $LOG_FILE && break

            echo "CORE 0 = $TEMP_CORE_0; CORE 1 = $TEMP_CORE_1; Duration: $COUNTER/$TEST_DURATION" | tee -a $LOG_FILE
            sleep $CHECK_FREQ
        done

        executeAsRoot "Turbo Boost on?" "echo 0 > /sys/devices/system/cpu/intel_pstate/no_turbo"
        getInfoAsRoot "cat /sys/devices/system/cpu/intel_pstate/no_turbo"
        executeAsRoot "Stopped benchmark?" 'kill $(pidof mprime)'
        executeAsRoot "Deleted benchmark files?" 'rm /usr/local/C15/scripts/prime.txt \
                                                 /usr/local/C15/scripts/local.txt  \
                                                 /usr/local/C15/scripts/results.txt'
        return 0
  
    else  # stress test with 'mprime' present in 'epcProductionHelpers.tar.gz' on an USB stick on the BBB, accessible via sshfs

        if ( getInfoAsRoot "[ -x /home/sscl/Prime95_Linux_19/mprime ]" ); then
            echo -e "${GREEN} benchmark program already present, skipping installation...${NC}"
        else
            while ! mount | grep /mnt/usb-stick > /dev/null || ! test -e /mnt/usb-stick/epcProductionHelpers.tar.gz ; do
                echo -e "${RED}Please connect a USB stick with 'epcProductionHelpers.tar.gz' to the BBB!${NC}"
                sleep 2
            done
        
            executeAsRoot \
                "Mount usb stick on ePC via sshfs?" \
                "sshfs -o allow_other,default_permissions root@192.168.10.11:/mnt/usb-stick/ /mnt/usb-stick"
            sleep 3
            BBB_STICK_MOUNTED=true
        
            executeAsRoot \
                 "Unpack 'epcProductionHelpers.tar.gz' from BBB USB stick to ePC?" \
                 "tar -xf /mnt/usb-stick/epcProductionHelpers.tar.gz -C /home/sscl"
      
      
            if ( ! getInfoAsRoot "[ -x /home/sscl/Prime95_Linux_19/mprime ]" ); then
                echo -e "${RED}'epcProductionHelpers.tar.gz' does not contain benchmarl program! Try again!${NC}" && read && exit 1
            fi
        fi

        executeAsRoot "Turbo Boost off?" "echo 1 > /sys/devices/system/cpu/intel_pstate/no_turbo"
        getInfoAsRoot "cat /sys/devices/system/cpu/intel_pstate/no_turbo"
        printf "Will run stress test on ePC for ${TEST_DURATION}s\n" | tee -a $LOG_FILE
        getInfoAsRoot "/home/sscl/Prime95_Linux_19/mprime -t &> /dev/null &"
        executeAsRoot "Benchmark started?" "pidof mprime"
        STRESS_TEST_STARTED=true

        for COUNTER in $(seq 0 $CHECK_FREQ $TEST_DURATION); do
            TEMP_CORE_0=$(getInfoAsRoot 'sensors | grep "Core 0:" | cut -d"+" -f2 | cut -d"." -f1')
            [ $TEMP_CORE_0 -ge 85 ] && printf "Core 0 critical: $TEMP_CORE_0\n" | tee -a $LOG_FILE && break

            TEMP_CORE_1=$(getInfoAsRoot 'sensors | grep "Core 1:" | cut -d"+" -f2 | cut -d"." -f1')
            [ $TEMP_CORE_1 -ge 85 ] && printf "Core 1 critical: $TEMP_CORE_1\n" | tee -a $LOG_FILE && break

            echo "CORE 0 = $TEMP_CORE_0; CORE 1 = $TEMP_CORE_1; Duration: $COUNTER/$TEST_DURATION" | tee -a $LOG_FILE
            sleep $CHECK_FREQ
        done

        executeAsRoot "Turbo Boost on?" "echo 0 > /sys/devices/system/cpu/intel_pstate/no_turbo"
        getInfoAsRoot "cat /sys/devices/system/cpu/intel_pstate/no_turbo"
        executeAsRoot "Stopped benchmark?" 'kill $(pidof mprime)'

        if [ "${BBB_STICK_MOUNTED}" == true ] ; then
            executeAsRoot \
                "Unmount usb stick on ePC via sshfs?" \
                "umount /mnt/usb-stick"
            BBB_STICK_MOUNTED=false
        fi

        return 0;
  
    fi
}

check_buffer_underruns() {
    TEST_DURATION=$1
    CHECK_FREQ=$2

    executeAsRoot \
        "Is audio-engine running?" \
        "systemctl status audio-engine" || return 0

    printf "Will run buffer underruns test on ePC for ${TEST_DURATION}s\n" | tee -a $LOG_FILE

    DATE=$(getInfoAsRoot 'date "+%y-%m-%d %T"')

    for COUNTER in $(seq 0 $CHECK_FREQ $TEST_DURATION); do
        if ( $(getInfoAsRoot "journalctl -u audio-engine -S \"${DATE}\" | grep -q x-run") ); then
            printf "${RED}Bufferunderun detected!${NC} " | tee -a $LOG_FILE \
                && getInfoAsRoot 'date "+%y-%m-%d %T"' | tee -a $LOG_FILE && return 1
        fi
        sleep $CHECK_FREQ;
    done
    return 0
}

usage() {
    echo "Runs a system check. I would advise to select a singular option!"
    echo
    echo "Syntax: ./NLc15-unit-check.sh [ -h ] [ -b DURATION,FACTOR ] [ -s|-S DURATION,FACTOR] [ -g ]"
    echo
    echo "options:"
    echo "-h          Usage output."
    echo "-b          Run buffer underruns test. DURATION of the test (Default 14400s), FACTOR will determine the output ratio (Default every 60s)."
    echo "-s          Run stress test from USB-stick on BBB. DURATION of the test (Default 1200s), FACTOR will determine the output ratio (Default every 20s)."
    echo "              Critical Temperature is set to 85C"
    echo "-S          same as -s but with benchmark pre-installed on ePC file system"
    echo "-g          Run general checks."
    echo
}

general_check() {
    printf "Will run general C15 checks for ePC, BBB and LPC\n\n" | tee -a $LOG_FILE
    set_path_variables

    check_userland_exists
    check_iface_names
    check_overlay
    check_soundcard_driver_param
    check_BBB_AP
    check_EPC_AP
    check_isolcpus
    check_mitigations
    check_ping_timeout

    get_date
    get_ro_mnts
    get_journal_dmesg
    get_partition_info

    check_persistent_integrity
    check_persistent_files
    check_alsa_settings

    get_network_settings_bbb
    get_update_script_vers
    check_for_present_tools
    get_uuids_and_info
    check_lpc
    check_usb_stick
}



cleanup() {
    echo -e "\ncleanup..."
    if [ "${BBB_STICK_MOUNTED}" == true ] ; then
        executeAsRoot \
            "Unmount usb stick on ePC via sshfs?" \
            "umount /mnt/usb-stick"
        BBB_STICK_MOUNTED=false
    fi

    if [ "${STRESS_TEST_STARTED}" == true ] ; then
        executeAsRoot "Stopped benchmark?" 'kill $(pidof mprime)'
        STRESS_TEST_STARTED=false
    fi

    exit 0
}

# ---- main script body ----
BBB_STICK_MOUNTED=false
STRESS_TEST_STARTED=false
trap cleanup EXIT

while getopts ":hgbsS" option; do
    case $option in
        b)
            set -f; OLDIFS=$IFS; IFS=','; NUMARR=($2); set +f; IFS=$OLDIFS;
            [ -z "${NUMARR[0]}" ] && DURATION=14400 || DURATION="${NUMARR[0]}"
            [ -z "${NUMARR[1]}" ] && FACTOR=60 || FACTOR="${NUMARR[1]}"
            LOG_FILE="/tmp/buffer_test.log"
            rm $LOG_FILE > /dev/null; touch $LOG_FILE
            check_buffer_underruns ${DURATION} ${FACTOR}
        ;;
        S|s)
            set -f; OLDIFS=$IFS; IFS=','; NUMARR=($2); set +f; IFS=$OLDIFS;
            LOG_FILE="/tmp/stress_test.log"
            rm $LOG_FILE > /dev/null; touch $LOG_FILE
            [ -z "${NUMARR[0]}" ] && DURATION=1200 || DURATION="${NUMARR[0]}"
            [ -z "${NUMARR[1]}" ] && FACTOR=20 || FACTOR="${NUMARR[1]}"
            run_epc_stress_test ${DURATION} ${FACTOR} ${option}
        ;;
        g)
            LOG_FILE="/tmp/general_check.log"
            rm $LOG_FILE > /dev/null; touch $LOG_FILE
            general_check
        ;;
        h | \?)
            usage
            exit
        ;;
        esac
done

[ $OPTIND -eq 1 ] && usage

